<?php

namespace App\Jobs;

use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DownloadProductImage implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $product;
    protected $imageUrls;

    /**
     * Create a new job instance.
     *
     * @param Product $product
     * @param array $imageUrls
     * @return void
     */
    public function __construct(Product $product, array $imageUrls)
    {
        $this->product = $product;
        $this->imageUrls = $imageUrls;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $product = $this->product;

        $savedImages = [];
        // Assuming 'images' is a JSON column in the database
        $existingImages = json_decode($product->images, true) ?? [];

        // Setup storage path
        $disk = 'public';
        $path = 'product/';

        // Create a hash map of existing images to avoid duplicates
        $existingImageHashMap = [];
        foreach ($existingImages as $existingImage) {
            // Extract filename without extension for comparison
            $filenameWithoutExt = pathinfo($existingImage, PATHINFO_FILENAME);
            // Remove random suffix if present (last 7 characters: dash + 6 random chars)
            if (strlen($filenameWithoutExt) > 7 && substr($filenameWithoutExt, -7, 1) === '-') {
                $baseFilename = substr($filenameWithoutExt, 0, -7);
                $existingImageHashMap[$baseFilename] = $existingImage;
            } else {
                $existingImageHashMap[$filenameWithoutExt] = $existingImage;
            }
        }

        foreach ($this->imageUrls as $index => $url) {
            try {
                // Generate a unique hash for the image URL to check for duplicates
                $urlHash = md5($url);
                
                // Check if this exact URL has been processed before by checking our custom log
                $imageLogPath = 'image_log/' . substr($urlHash, 0, 2) . '/' . $urlHash . '.log';
                if (Storage::disk($disk)->exists($imageLogPath)) {
                    // Image already processed, get the saved filename
                    $savedFilename = Storage::disk($disk)->get($imageLogPath);
                    $savedImages[] = trim($savedFilename);
                    
                    // Set first image as thumbnail if empty
                    if ($index === 0 && empty($product->thumbnail)) {
                        $product->thumbnail = trim($savedFilename);
                    }
                    continue;
                }

                $response = Http::timeout(30)->get($url);

                if ($response->successful()) {
                    $extension = pathinfo($url, PATHINFO_EXTENSION);
                    if (!$extension) $extension = 'jpg'; // Default fallback

                    // Clean extension for query params
                    $extension = explode('?', $extension)[0];
                    
                    // Generate base filename from product slug
                    $baseFilename = $product->slug ?: ('product-' . $product->id);
                    
                    // Create a unique filename with random suffix
                    $filename = $baseFilename . '-' . Str::random(6) . '.' . $extension;
                    $fullPath = $path . $filename;

                    // Save the image content
                    Storage::disk($disk)->put($fullPath, $response->body());

                    // Log this URL to prevent future duplicate downloads
                    Storage::disk($disk)->put($imageLogPath, $filename);

                    $savedImages[] = $filename;

                    // Set first image as thumbnail if empty
                    if ($index === 0 && empty($product->thumbnail)) {
                        $product->thumbnail = $filename;
                    }
                }
            } catch (\Exception $e) {
                // Log error but continue with other images
                \Log::error("Failed to download image for Product {$this->product->id}: " . $url . " Error: " . $e->getMessage());
            }
        }

        // Merge new images with existing, avoiding duplicates
        $finalImages = array_values(array_unique(array_merge($existingImages, $savedImages)));

        $product->images = json_encode($finalImages);
        $product->save();
    }
}